/**
 * auth.ts
 * Centralized Authentication Service for Indzs Account
 */

const BASE_URL = process.env.NEXT_PUBLIC_API_BASE_URL;

// --- Interfaces ---

export interface User {
    id: string;
    first_name: string;
    last_name?: string;
    email: string;
    phone_number: string;
    isVerified: boolean;
    primaryPhoneVerified?: boolean,
    recovery_phone?: string,
    recovery_phone_verified?: boolean,
    primary_email?: string,
    primary_email_verified?: boolean, // User needs to verify
    recovery_email?: string,
    recovery_email_verified?: boolean,
    dob?: string,
    gender?: string
    profile_picture?: string,
    address?:string,
    connected_apps?:[]
}

export interface Session {
    deviceId: string;
    deviceInfo: string;
    lastActive: string;
    isCurrentDevice: boolean;
}

export interface AuthResponse<T = any> {
    success: boolean;
    message?: string;
    data?: T;
    unverified?: boolean; // Specific flag for login flow
}

// --- Device Helpers ---

const getDeviceInfo = (): string => {
    if (typeof window === "undefined") return "Server Side";
    return `Platform: ${navigator.platform}, UA: ${navigator.userAgent}`;
};

const getOrCreateDeviceId = (): string => {
    if (typeof window === "undefined") return "";
    let id = localStorage.getItem("device_id");
    if (!id) {
        id = crypto.randomUUID?.() || Math.random().toString(36).substring(2);
        localStorage.setItem("device_id", id);
    }
    return id;
};


// --- State for Refresh Queueing ---
let isRefreshing = false;
let refreshSubscribers: ((token: string) => void)[] = [];

const subscribeTokenRefresh = (cb: (token: string) => void) => {
    refreshSubscribers.push(cb);
};

const onRerefreshSuccess = (token: string) => {
    refreshSubscribers.map((cb) => cb(token));
    refreshSubscribers = [];
};


// --- API Service ---

const apiFetch = async (endpoint: string, options: RequestInit = {}) => {
    const headers = new Headers(options.headers);
    headers.set("Content-Type", "application/json");

    if (typeof window !== "undefined") {
        headers.set("X-Device-Id", getOrCreateDeviceId());
        headers.set("X-Device-Info", getDeviceInfo());
    }

    const config = {
        ...options,
        headers,
        credentials: "include" as RequestCredentials, // Required for HttpOnly cookies
    };

    const response = await fetch(`${BASE_URL}${endpoint}`, config);
    return response;
};



export const authService = {
    // 1. REGISTRATION
    async initiateRegister(data: Partial<User> & { password?: string }): Promise<AuthResponse> {
        const res = await apiFetch("/api/auth/register/", {
            method: "POST",
            body: JSON.stringify(data),
        });
        return res.json();
    },

    async verifyRegisterOtp(phone: string, otp: string): Promise<AuthResponse> {
        const res = await apiFetch("/api/auth/register/verify/", {
            method: "POST",
            body: JSON.stringify({ phone, otp }),
        });
        return res.json();
    },


    async refreshToken(): Promise<AuthResponse<{ accessToken: string }>> {
        try {
            const res = await fetch(`${BASE_URL}/api/auth/refresh/`, {
                method: "POST",
                headers: {
                    "Content-Type": "application/json",
                    "X-Device-Id": localStorage.getItem("device_id") || "",
                },
                credentials: "include", // Essential for HttpOnly refresh cookies
            });

            const data = await res.json();
            if (!res.ok) throw new Error("Session expired");

            return { success: true, data };
        } catch (error) {
            return { success: false, message: "Session expired" };
        }
    },

    /**
     * AUTHENTICATED FETCH WRAPPER
     * Intercepts 401 errors to attempt a refresh before failing.
     */
    async authenticatedFetch(endpoint: string, options: RequestInit = {}): Promise<Response> {
        const url = `${BASE_URL}${endpoint}`;

        // Standard headers
        const headers = new Headers(options.headers);
        headers.set("Content-Type", "application/json");
        if (typeof window !== "undefined") {
            headers.set("X-Device-Id", localStorage.getItem("device_id") || "");
        }

        const config = { ...options, headers, credentials: "include" as RequestCredentials };

        const response = await fetch(url, config);
        console.log(response)
        // If unauthorized (expired access token)
        if (response.status === 401) {
            if (!isRefreshing) {
                isRefreshing = true;

                const refreshResult = await this.refreshToken();

                if (refreshResult.success) {
                    isRefreshing = false;
                    onRerefreshSuccess(refreshResult.data?.accessToken || "");
                    // Retry the original request
                    return fetch(url, config);
                } else {
                    isRefreshing = false;
                    // Optional: redirect to login
                    // window.location.href = "/login";
                    return response
                }
            }

            // If a refresh is already in progress, wait for it
            return new Promise((resolve) => {
                subscribeTokenRefresh((newToken) => {
                    // Update headers if using Bearer token, then retry
                    // if using Cookies, just retry
                    resolve(fetch(url, config));
                });
            });
        }

        return response;
    },



    // 2. LOGIN (Returns raw response to handle 403 Unverified status)
    async login(identifier: string, password: string): Promise<Response> {
        const device_id = getOrCreateDeviceId()
        return authService.authenticatedFetch("/api/auth/login/", {
            method: "POST",
            body: JSON.stringify({ identifier, password, "X-Device-Id":device_id  }),
        });
    },

    // 3. USER INFO & SESSIONS
    async getUserInfo(): Promise<AuthResponse<User>> {
        const res = await authService.authenticatedFetch("/api/auth/me/");
        // console.log(await res.json())
        return res.json();
    },

    async getActiveSessions(): Promise<AuthResponse<Session[]>> {
        const res = await apiFetch("/api/auth/sessions/");
        return res.json();
    },

    // 4. LOGOUT LOGIC
    async logout(): Promise<AuthResponse> {
        const res = await apiFetch("/api/auth/logout/", { method: "POST" });
        return res.json();
    },

    async logoutAllDevices(): Promise<AuthResponse> {
        const res = await apiFetch("/api/auth/logout-all/", { method: "POST" });
        return res.json();
    },

    async logoutSpecificDevice(deviceId: string): Promise<AuthResponse> {
        const res = await apiFetch("/api/auth/logout-device/", {
            method: "POST",
            body: JSON.stringify({ deviceId }),
        });
        return res.json();
    },

    // 5. RECOVERY
    async forgotPassword(email: string): Promise<AuthResponse> {
        const res = await apiFetch("/api/auth/forgot-password/", {
            method: "POST",
            body: JSON.stringify({ email }),
        });
        return res.json();
    },

    async forgotAccount(phone: string): Promise<AuthResponse> {
        const res = await apiFetch("/api/auth/forgot-account/", {
            method: "POST",
            body: JSON.stringify({ phone }),
        });
        return res.json();
    }
};